# 機能設計書 92-集合演算（Set Operations）

## 概要

本ドキュメントは、TensorFlowにおける集合演算（Set Operations）機能の設計を記述する。テンソル間の集合和・積・差などの集合演算を提供する。

### 本機能の処理概要

**業務上の目的・背景**：機械学習や推薦システムにおいて、集合間の関係を計算する必要がある場面がある。例えば、ユーザの過去の購入商品集合と推薦候補集合の共通部分（intersection）、新規推薦の差分（difference）、全候補の合計（union）などである。本機能は、密テンソルやスパーステンソルに対する集合演算をC++カーネルレベルで効率的に実装する。

**機能の利用シーン**：推薦システムでの候補フィルタリング、テキスト処理での語彙集合操作、データ前処理での重複除去・共通要素抽出などに利用される。

**主要な処理内容**：
1. `SetSize` - スパーステンソルで表された集合のサイズ（要素数）を計算
2. `DenseToDenseSetOperation` - 2つの密テンソル間の集合演算（union, intersection, difference）
3. `DenseToSparseSetOperation` - 密テンソルとスパーステンソル間の集合演算
4. `SparseToSparseSetOperation` - 2つのスパーステンソル間の集合演算

**関連システム・外部連携**：SparseTensor演算（sparse_ops）と連携。結果はSparseTensor形式で返される。

**権限による制御**：特になし。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能はAPI機能であり、直接関連する画面はない |

## 機能種別

計算処理（集合演算）

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| set1 / set2 | Tensor (Dense) | Yes（Dense系） | 集合を表す密テンソル | rank >= 2 |
| set_indices | int64 | Yes（Sparse系） | SparseTensorのインデックス | 有効なスパースインデックス |
| set_values | T | Yes | 集合の要素値 | T: int8, int16, int32, int64, uint8, uint16, string |
| set_shape | int64 | Yes（Sparse系） | SparseTensorの形状 | 有効な形状 |
| set_operation | string | Yes | 集合演算種別 | "a-b", "b-a", "intersection", "union" 等 |
| validate_indices | bool | No | インデックスの検証有無（デフォルト: true） | - |

### 入力データソース

計算グラフ内のテンソル（密テンソルまたはSparseTensor）。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| result_indices | int64 | 結果SparseTensorのインデックス |
| result_values | T | 結果SparseTensorの値 |
| result_shape | int64 | 結果SparseTensorの形状 |
| size | int32 | 集合サイズ（SetSizeのみ） |

### 出力先

計算グラフ内の後続操作への入力。SparseTensor形式で返却。

## 処理フロー

### 処理シーケンス

```
1. 入力テンソルの形状検証
   └─ rank >= 2 の確認、入力間のrank一致確認
2. SparseTensor有効性検証（Sparse入力の場合）
   └─ ValidateSparseTensorによるインデックス・形状整合性チェック
3. 集合演算の実行
   └─ set_operation属性に基づき union / intersection / difference を計算
4. 結果のSparseTensor構築
   └─ result_indices, result_values, result_shape の生成
```

### フローチャート

```mermaid
flowchart TD
    A[入力: set1, set2] --> B{入力形式?}
    B -->|Dense-Dense| C[DenseToDenseSetOperation]
    B -->|Dense-Sparse| D[DenseToSparseSetOperation]
    B -->|Sparse-Sparse| E[SparseToSparseSetOperation]
    C --> F[rank検証: >= 2, 一致]
    D --> G[rank検証 + SparseTensor検証]
    E --> H[SparseTensor検証 x 2]
    F --> I[集合演算実行]
    G --> I
    H --> I
    I --> J[結果SparseTensor生成]
    J --> K[result_indices, result_values, result_shape]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-92-01 | rank制約 | 入力テンソルのrankは2以上 | 全演算 |
| BR-92-02 | rank一致 | 2つの入力テンソルのrankは一致する必要がある | 全演算 |
| BR-92-03 | グループ形状一致 | 最後の次元を除く先頭 n-1 次元は互換性がある必要がある | DenseToDense |
| BR-92-04 | 型制約 | サポート型: int8, int16, int32, int64, uint8, uint16, string | 全演算 |

### 計算ロジック

集合演算の種別：
- `union`: 和集合（A ∪ B）
- `intersection`: 積集合（A ∩ B）
- `a-b`: 差集合（A - B）
- `b-a`: 差集合（B - A）

## データベース操作仕様

### 操作別データベース影響一覧

本機能はデータベース操作を行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| InvalidArgument | 入力数不正 | DenseToDenseで入力数が2でない | 正しい入力数を指定 |
| InvalidArgument | rank不足 | 入力テンソルのrankが2未満 | rank >= 2のテンソルを入力 |
| InvalidArgument | rank不一致 | 2つの入力テンソルのrankが異なる | 同じrankのテンソルを入力 |
| InvalidArgument | 形状不一致 | 先頭n-1次元が互換でない | 互換性のある形状を使用 |

### リトライ仕様

リトライ不要。形状エラーは即時送出。

## トランザクション仕様

該当なし。ステートレスなテンソル演算。

## パフォーマンス要件

- 集合演算はC++カーネルで実行されるため高速
- 大規模なSparseTensorに対しても効率的に動作

## セキュリティ考慮事項

- validate_indices=trueでインデックスの妥当性を検証（デフォルト有効）

## 備考

- 最後の次元が集合要素を表し、それ以外の次元はバッチ/グループ次元として扱われる
- 結果は常にSparseTensor形式（indices, values, shape）で返される

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: Op定義を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | set_ops.cc | `tensorflow/core/ops/set_ops.cc` | 4つのOp登録（SetSize, DenseToDense, DenseToSparse, SparseToSparse） |

**主要処理フロー**:
1. **26-33行目**: `SetSize` Op - SparseTensorの集合サイズを計算
2. **35-89行目**: `DenseToDenseSetOperation` Op - 密テンソル間の集合演算、形状推論あり
3. **91-133行目**: `DenseToSparseSetOperation` Op - 密/スパース混合の集合演算
4. **135-189行目**: `SparseToSparseSetOperation` Op - スパーステンソル間の集合演算

**読解のコツ**: 各Opの`SetShapeFn`ラムダで形状推論ロジックが定義されている。出力は常に3つ（indices, values, shape）のSparseTensor構成要素。rank検証の違いに注目。

#### Step 2: カーネル実装を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | set_kernels.cc | `tensorflow/core/kernels/set_kernels.cc` | 実際の集合演算アルゴリズム実装 |

### プログラム呼び出し階層図

```
tf.sets.intersection / union / difference (Python API)
    │
    ├─ DenseToDenseSetOperation (密-密)
    │      └─ SetShapeFn: rank検証 + 形状互換性チェック
    │
    ├─ DenseToSparseSetOperation (密-スパース)
    │      └─ SetShapeFn: ValidateSparseTensor + rank検証
    │
    ├─ SparseToSparseSetOperation (スパース-スパース)
    │      └─ SetShapeFn: ValidateSparseTensor x 2
    │
    └─ SetSize (サイズ計算)
           └─ SetShapeFn: UnknownShape
```

### データフロー図

```
[入力]                          [処理]                    [出力]

Dense Tensor set1    ───▶  SetOperation Kernel    ───▶  SparseTensor
Dense Tensor set2    ───▶   (union/intersection/        ├─ result_indices
  or                         difference)                ├─ result_values
SparseTensor set2    ───▶                               └─ result_shape
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| set_ops.cc | `tensorflow/core/ops/set_ops.cc` | ソース | Op定義と形状推論 |
| set_kernels.cc | `tensorflow/core/kernels/set_kernels.cc` | ソース | C++カーネル実装 |
| sets.py | `tensorflow/python/ops/sets.py` | ソース | Python API |
